/**
 * \file: ipu_dp_csc_demo.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * Test application to update IPU display processor color space (CSC) matrix by
 * passing DRM handle and color property values.
 *
 * \component: IPU
 *
 * \author: Gautham Kantharaju / RBEI / ECF3 / Gautham.Kantharaju@in.bosch.com
 *
 * \copyright: (c) 2015 ADIT Corporation
 *
 * \history
 * 0.1 Gautham Kantharaju Initial version
 ******************************************************************************/
#include <ipu_dp_csc.h>
#include <unistd.h>

//#define DEBUG

static const char device_name[] = "/dev/dri/card0";

static void usage(char *argv[])
{
	fprintf(stderr, "Usage: %s <csc_update>"
			" <hue> <saturation> <brightness> <contrast>"
			" <hue_off> <saturation_off> <brightness_off>\n", argv[0]);

	exit(EXIT_FAILURE);
}

int main(int argc, char *argv[])
{
	CLRPROP cprop;
	int drm_fd = -1;
	bool csc_update = false;
	int i = 0;
	int j = 0;
	int res = 0;
	uint32_t crtc_id;
	CLRSPACETYPE type;

	if (argc != 9)
		usage(argv);

	/*
	 * Pass cprop structure in turn library
	 * will give back default color property
	 * value and CSC identity matrix
	 */
	csc_set_default_value(&cprop);

#ifdef DEBUG
	fprintf (stdout, "Appln Hue = %d \n"
			"Saturation = %d \n"
			"Brightness = %d \n"
			"Contrast = %d \n"
			"Hue offset = %d \n"
			"Saturation offset = %d \n"
			"Brightness offset = %d\n",
			cprop.hue, cprop.saturation, cprop.brightness, cprop.contrast,
			cprop.hue_off, cprop.saturation_off, cprop.brightness_off);
	/* Printing the computed CSC matrix */
	for (i = 0; i < 3; i++)
	{
		for (j = 0; j < 3; j++)
		{
			fprintf (stdout, "Appln A[%d][%d] = %d \t", i, j, cprop.csc.A[i][j]);
		}

		printf ("\n");
	}

	for (i = 0; i < 3; i++)
		fprintf (stdout, "\nAppln B[%d]= %d \t", i, cprop.csc.B[i]);

	printf ("\n");

	for (i = 0; i < 3; i++)
		fprintf (stdout, "\nAppln E[%d] = %d \t", i, cprop.csc.E[i]);

	printf ("\n");
#endif
	if(!strcasecmp(argv[1], "true"))
		csc_update = true;
	else
		csc_update = false;

	cprop.hue = strtol(argv[2], NULL, 0);
	cprop.saturation = strtol(argv[3], NULL, 0);
	cprop.brightness = strtol(argv[4], NULL, 0);
	cprop.contrast = strtol(argv[5], NULL, 0);

	cprop.hue_off = strtol(argv[6], NULL, 0);
	cprop.saturation_off = strtol(argv[7], NULL, 0);
	cprop.brightness_off = strtol(argv[8], NULL, 0);

	drm_fd = open(device_name, (O_RDWR | O_CLOEXEC));
	if (drm_fd < 0)
	{
		perror("cannot open drm device");
		exit(EXIT_FAILURE);
	}

	/* Update drm_imx_csc */

	/*
	 * By default crtc id will be retrieved by library interface
	 * (libipu_dp_csc.so) if crtc id passed with value "0xFF" for
	 * the first time and also supported crtc id's will be provided
	 * to application else library will use the crtc id passed by the
	 * application.
	 */
	crtc_id = INVALID_VALUE;
	csc_update_crtc_id (crtc_id);

	/*
	 * By default CSC type will be "DRM_IMX_CSC_RGB2RGB"
	 * if type passed with value "0xFF" else library
	 * will use type passed by the application.
	 */
	type = (CLRSPACETYPE) INVALID_VALUE;
	csc_update_color_space_conversion_type (type);

	res = csc_dp_compute_update_matrix (drm_fd, csc_update, &cprop);
	if (res < 0)
	{
		fprintf (stdout, "Could not update CSC matrix!!! \n");
		exit (EXIT_FAILURE);
	}

	sleep(2);

	/*
	 * Next time onwards pass supported crtc id i.e.
	 * for ex: For target boards like i.MX6 [Solo|Dual-Lite]
	 * IPU unit is one and it supports two crtc's out of which
	 * only one will support update of CSC matrix i.e. supported_crtc_id[0].
	 * For i.MX6 Quad boards IPU unit is two and it supports
	 * four crtc's out of which only two will support update of
	 * CSC matrix i.e. supported_crtc_id[0] and supported_crtc_id[1]
	 */
	crtc_id = cprop.supported_crtc_id[0];
	csc_update_crtc_id (crtc_id);

	cprop.hue = 0xFF;
	cprop.saturation = 0xFF;
	cprop.brightness = 0xFF;
	cprop.contrast = 0xFF;
	cprop.hue_off = 0xFF;
	cprop.saturation_off = 50;
	cprop.brightness_off = 0xFF;

	res = csc_dp_compute_update_matrix (drm_fd, csc_update, &cprop);
	if (res < 0)
	{
		fprintf (stdout, "Could not update CSC matrix!!! \n");
		exit (EXIT_FAILURE);
	}

	sleep(2);

	crtc_id = cprop.supported_crtc_id[0];
	csc_update_crtc_id (crtc_id);

	cprop.hue = 0xFF;
	cprop.saturation = 0xFF;
	cprop.brightness = 0xFF;
	cprop.contrast = 0xFF;
	cprop.hue_off = 120;
	cprop.saturation_off = 0xFF;
	cprop.brightness_off = 0xFF;
	res = csc_dp_compute_update_matrix (drm_fd, csc_update, &cprop);

	if (res < 0)
	{
		fprintf (stdout, "Could not update CSC matrix!!! \n");
		exit (EXIT_FAILURE);
	}
	else
	{
		/* Printing the latest computed CSC matrix */
		for (i = 0; i < 3; i++)
		{
			for (j = 0; j < 3; j++)
			{
				fprintf (stdout, "A[%d][%d] = %d \t", i, j, cprop.csc.A[i][j]);
			}

			printf ("\n");
		}

		for (i = 0; i < 3; i++)
			fprintf (stdout, "\nB[%d]= %d \t", i, cprop.csc.B[i]);

		printf ("\n");

		for (i = 0; i < 3; i++)
			fprintf (stdout, "\nE[%d] = %d \t", i, cprop.csc.E[i]);

		printf ("\n");
	}

	sleep(2);

	/*
	 * Atlast update CSC with identity matrix
	 * and disable the CSC unit
	 */
	/* Set default color property values */
	csc_set_default_value(&cprop);

	crtc_id = cprop.supported_crtc_id[0];
	csc_update_crtc_id (crtc_id);

	/* Compute and update CSC with identity matrix */
	res = csc_dp_compute_update_matrix (drm_fd, csc_update, &cprop);

	sleep(2);

	/* Disable CSC unit */
	cprop.csc.enable = false;
	res = drmIoctl(drm_fd, DRM_IOCTL_IMX_SET_CSC, &cprop.csc);
	if (res)
	{
		fprintf(stdout, "failed to disable CSC unit\n");
		exit (EXIT_FAILURE);
	}

	close (drm_fd);

	exit (EXIT_SUCCESS);
}
